/*
 * Copyright (C) 2015 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "SdkConstants.h"

#include <algorithm>
#include <string>
#include <unordered_map>
#include <vector>

using android::StringPiece;

namespace aapt {

static const char* sDevelopmentSdkCodeName = "O";
static int sDevelopmentSdkLevel = 26;

static const std::vector<std::pair<uint16_t, size_t>> sAttrIdMap = {
    {0x021c, 1},
    {0x021d, 2},
    {0x0269, SDK_CUPCAKE},
    {0x028d, SDK_DONUT},
    {0x02ad, SDK_ECLAIR},
    {0x02b3, SDK_ECLAIR_0_1},
    {0x02b5, SDK_ECLAIR_MR1},
    {0x02bd, SDK_FROYO},
    {0x02cb, SDK_GINGERBREAD},
    {0x0361, SDK_HONEYCOMB},
    {0x0363, SDK_HONEYCOMB_MR1},
    {0x0366, SDK_HONEYCOMB_MR2},
    {0x03a6, SDK_ICE_CREAM_SANDWICH},
    {0x03ae, SDK_JELLY_BEAN},
    {0x03cc, SDK_JELLY_BEAN_MR1},
    {0x03da, SDK_JELLY_BEAN_MR2},
    {0x03f1, SDK_KITKAT},
    {0x03f6, SDK_KITKAT_WATCH},
    {0x04ce, SDK_LOLLIPOP},
};

static bool less_entry_id(const std::pair<uint16_t, size_t>& p,
                        uint16_t entryId) {
  return p.first < entryId;
}

size_t FindAttributeSdkLevel(const ResourceId& id) {
  if (id.package_id() != 0x01 && id.type_id() != 0x01) {
    return 0;
  }
  auto iter = std::lower_bound(sAttrIdMap.begin(), sAttrIdMap.end(),
                               id.entry_id(), less_entry_id);
  if (iter == sAttrIdMap.end()) {
    return SDK_LOLLIPOP_MR1;
  }
  return iter->second;
}

static const std::unordered_map<std::string, size_t> sAttrMap = {
    {"marqueeRepeatLimit", 2},
    {"windowNoDisplay", 3},
    {"backgroundDimEnabled", 3},
    {"inputType", 3},
    {"isDefault", 3},
    {"windowDisablePreview", 3},
    {"privateImeOptions", 3},
    {"editorExtras", 3},
    {"settingsActivity", 3},
    {"fastScrollEnabled", 3},
    {"reqTouchScreen", 3},
    {"reqKeyboardType", 3},
    {"reqHardKeyboard", 3},
    {"reqNavigation", 3},
    {"windowSoftInputMode", 3},
    {"imeFullscreenBackground", 3},
    {"noHistory", 3},
    {"headerDividersEnabled", 3},
    {"footerDividersEnabled", 3},
    {"candidatesTextStyleSpans", 3},
    {"smoothScrollbar", 3},
    {"reqFiveWayNav", 3},
    {"keyBackground", 3},
    {"keyTextSize", 3},
    {"labelTextSize", 3},
    {"keyTextColor", 3},
    {"keyPreviewLayout", 3},
    {"keyPreviewOffset", 3},
    {"keyPreviewHeight", 3},
    {"verticalCorrection", 3},
    {"popupLayout", 3},
    {"state_long_pressable", 3},
    {"keyWidth", 3},
    {"keyHeight", 3},
    {"horizontalGap", 3},
    {"verticalGap", 3},
    {"rowEdgeFlags", 3},
    {"codes", 3},
    {"popupKeyboard", 3},
    {"popupCharacters", 3},
    {"keyEdgeFlags", 3},
    {"isModifier", 3},
    {"isSticky", 3},
    {"isRepeatable", 3},
    {"iconPreview", 3},
    {"keyOutputText", 3},
    {"keyLabel", 3},
    {"keyIcon", 3},
    {"keyboardMode", 3},
    {"isScrollContainer", 3},
    {"fillEnabled", 3},
    {"updatePeriodMillis", 3},
    {"initialLayout", 3},
    {"voiceSearchMode", 3},
    {"voiceLanguageModel", 3},
    {"voicePromptText", 3},
    {"voiceLanguage", 3},
    {"voiceMaxResults", 3},
    {"bottomOffset", 3},
    {"topOffset", 3},
    {"allowSingleTap", 3},
    {"handle", 3},
    {"content", 3},
    {"animateOnClick", 3},
    {"configure", 3},
    {"hapticFeedbackEnabled", 3},
    {"innerRadius", 3},
    {"thickness", 3},
    {"sharedUserLabel", 3},
    {"dropDownWidth", 3},
    {"dropDownAnchor", 3},
    {"imeOptions", 3},
    {"imeActionLabel", 3},
    {"imeActionId", 3},
    {"imeExtractEnterAnimation", 3},
    {"imeExtractExitAnimation", 3},
    {"tension", 4},
    {"extraTension", 4},
    {"anyDensity", 4},
    {"searchSuggestThreshold", 4},
    {"includeInGlobalSearch", 4},
    {"onClick", 4},
    {"targetSdkVersion", 4},
    {"maxSdkVersion", 4},
    {"testOnly", 4},
    {"contentDescription", 4},
    {"gestureStrokeWidth", 4},
    {"gestureColor", 4},
    {"uncertainGestureColor", 4},
    {"fadeOffset", 4},
    {"fadeDuration", 4},
    {"gestureStrokeType", 4},
    {"gestureStrokeLengthThreshold", 4},
    {"gestureStrokeSquarenessThreshold", 4},
    {"gestureStrokeAngleThreshold", 4},
    {"eventsInterceptionEnabled", 4},
    {"fadeEnabled", 4},
    {"backupAgent", 4},
    {"allowBackup", 4},
    {"glEsVersion", 4},
    {"queryAfterZeroResults", 4},
    {"dropDownHeight", 4},
    {"smallScreens", 4},
    {"normalScreens", 4},
    {"largeScreens", 4},
    {"progressBarStyleInverse", 4},
    {"progressBarStyleSmallInverse", 4},
    {"progressBarStyleLargeInverse", 4},
    {"searchSettingsDescription", 4},
    {"textColorPrimaryInverseDisableOnly", 4},
    {"autoUrlDetect", 4},
    {"resizeable", 4},
    {"required", 5},
    {"accountType", 5},
    {"contentAuthority", 5},
    {"userVisible", 5},
    {"windowShowWallpaper", 5},
    {"wallpaperOpenEnterAnimation", 5},
    {"wallpaperOpenExitAnimation", 5},
    {"wallpaperCloseEnterAnimation", 5},
    {"wallpaperCloseExitAnimation", 5},
    {"wallpaperIntraOpenEnterAnimation", 5},
    {"wallpaperIntraOpenExitAnimation", 5},
    {"wallpaperIntraCloseEnterAnimation", 5},
    {"wallpaperIntraCloseExitAnimation", 5},
    {"supportsUploading", 5},
    {"killAfterRestore", 5},
    {"restoreNeedsApplication", 5},
    {"smallIcon", 5},
    {"accountPreferences", 5},
    {"textAppearanceSearchResultSubtitle", 5},
    {"textAppearanceSearchResultTitle", 5},
    {"summaryColumn", 5},
    {"detailColumn", 5},
    {"detailSocialSummary", 5},
    {"thumbnail", 5},
    {"detachWallpaper", 5},
    {"finishOnCloseSystemDialogs", 5},
    {"scrollbarFadeDuration", 5},
    {"scrollbarDefaultDelayBeforeFade", 5},
    {"fadeScrollbars", 5},
    {"colorBackgroundCacheHint", 5},
    {"dropDownHorizontalOffset", 5},
    {"dropDownVerticalOffset", 5},
    {"quickContactBadgeStyleWindowSmall", 6},
    {"quickContactBadgeStyleWindowMedium", 6},
    {"quickContactBadgeStyleWindowLarge", 6},
    {"quickContactBadgeStyleSmallWindowSmall", 6},
    {"quickContactBadgeStyleSmallWindowMedium", 6},
    {"quickContactBadgeStyleSmallWindowLarge", 6},
    {"author", 7},
    {"autoStart", 7},
    {"expandableListViewWhiteStyle", 8},
    {"installLocation", 8},
    {"vmSafeMode", 8},
    {"webTextViewStyle", 8},
    {"restoreAnyVersion", 8},
    {"tabStripLeft", 8},
    {"tabStripRight", 8},
    {"tabStripEnabled", 8},
    {"logo", 9},
    {"xlargeScreens", 9},
    {"immersive", 9},
    {"overScrollMode", 9},
    {"overScrollHeader", 9},
    {"overScrollFooter", 9},
    {"filterTouchesWhenObscured", 9},
    {"textSelectHandleLeft", 9},
    {"textSelectHandleRight", 9},
    {"textSelectHandle", 9},
    {"textSelectHandleWindowStyle", 9},
    {"popupAnimationStyle", 9},
    {"screenSize", 9},
    {"screenDensity", 9},
    {"allContactsName", 11},
    {"windowActionBar", 11},
    {"actionBarStyle", 11},
    {"navigationMode", 11},
    {"displayOptions", 11},
    {"subtitle", 11},
    {"customNavigationLayout", 11},
    {"hardwareAccelerated", 11},
    {"measureWithLargestChild", 11},
    {"animateFirstView", 11},
    {"dropDownSpinnerStyle", 11},
    {"actionDropDownStyle", 11},
    {"actionButtonStyle", 11},
    {"showAsAction", 11},
    {"previewImage", 11},
    {"actionModeBackground", 11},
    {"actionModeCloseDrawable", 11},
    {"windowActionModeOverlay", 11},
    {"valueFrom", 11},
    {"valueTo", 11},
    {"valueType", 11},
    {"propertyName", 11},
    {"ordering", 11},
    {"fragment", 11},
    {"windowActionBarOverlay", 11},
    {"fragmentOpenEnterAnimation", 11},
    {"fragmentOpenExitAnimation", 11},
    {"fragmentCloseEnterAnimation", 11},
    {"fragmentCloseExitAnimation", 11},
    {"fragmentFadeEnterAnimation", 11},
    {"fragmentFadeExitAnimation", 11},
    {"actionBarSize", 11},
    {"imeSubtypeLocale", 11},
    {"imeSubtypeMode", 11},
    {"imeSubtypeExtraValue", 11},
    {"splitMotionEvents", 11},
    {"listChoiceBackgroundIndicator", 11},
    {"spinnerMode", 11},
    {"animateLayoutChanges", 11},
    {"actionBarTabStyle", 11},
    {"actionBarTabBarStyle", 11},
    {"actionBarTabTextStyle", 11},
    {"actionOverflowButtonStyle", 11},
    {"actionModeCloseButtonStyle", 11},
    {"titleTextStyle", 11},
    {"subtitleTextStyle", 11},
    {"iconifiedByDefault", 11},
    {"actionLayout", 11},
    {"actionViewClass", 11},
    {"activatedBackgroundIndicator", 11},
    {"state_activated", 11},
    {"listPopupWindowStyle", 11},
    {"popupMenuStyle", 11},
    {"textAppearanceLargePopupMen", 11},
    {"textAppearanceSmallPopupMen", 11},
    {"breadCrumbTitle", 11},
    {"breadCrumbShortTitle", 11},
    {"listDividerAlertDialog", 11},
    {"textColorAlertDialogListItem", 11},
    {"loopViews", 11},
    {"dialogTheme", 11},
    {"alertDialogTheme", 11},
    {"dividerVertical", 11},
    {"homeAsUpIndicator", 11},
    {"enterFadeDuration", 11},
    {"exitFadeDuration", 11},
    {"selectableItemBackground", 11},
    {"autoAdvanceViewId", 11},
    {"useIntrinsicSizeAsMinimum", 11},
    {"actionModeCutDrawable", 11},
    {"actionModeCopyDrawable", 11},
    {"actionModePasteDrawable", 11},
    {"textEditPasteWindowLayout", 11},
    {"textEditNoPasteWindowLayout", 11},
    {"textIsSelectable", 11},
    {"windowEnableSplitTouch", 11},
    {"indeterminateProgressStyle", 11},
    {"progressBarPadding", 11},
    {"animationResolution", 11},
    {"state_accelerated", 11},
    {"baseline", 11},
    {"homeLayout", 11},
    {"opacity", 11},
    {"alpha", 11},
    {"transformPivotX", 11},
    {"transformPivotY", 11},
    {"translationX", 11},
    {"translationY", 11},
    {"scaleX", 11},
    {"scaleY", 11},
    {"rotation", 11},
    {"rotationX", 11},
    {"rotationY", 11},
    {"showDividers", 11},
    {"dividerPadding", 11},
    {"borderlessButtonStyle", 11},
    {"dividerHorizontal", 11},
    {"itemPadding", 11},
    {"buttonBarStyle", 11},
    {"buttonBarButtonStyle", 11},
    {"segmentedButtonStyle", 11},
    {"staticWallpaperPreview", 11},
    {"allowParallelSyncs", 11},
    {"isAlwaysSyncable", 11},
    {"verticalScrollbarPosition", 11},
    {"fastScrollAlwaysVisible", 11},
    {"fastScrollThumbDrawable", 11},
    {"fastScrollPreviewBackgroundLeft", 11},
    {"fastScrollPreviewBackgroundRight", 11},
    {"fastScrollTrackDrawable", 11},
    {"fastScrollOverlayPosition", 11},
    {"customTokens", 11},
    {"nextFocusForward", 11},
    {"firstDayOfWeek", 11},
    {"showWeekNumber", 11},
    {"minDate", 11},
    {"maxDate", 11},
    {"shownWeekCount", 11},
    {"selectedWeekBackgroundColor", 11},
    {"focusedMonthDateColor", 11},
    {"unfocusedMonthDateColor", 11},
    {"weekNumberColor", 11},
    {"weekSeparatorLineColor", 11},
    {"selectedDateVerticalBar", 11},
    {"weekDayTextAppearance", 11},
    {"dateTextAppearance", 11},
    {"solidColor", 11},
    {"spinnersShown", 11},
    {"calendarViewShown", 11},
    {"state_multiline", 11},
    {"detailsElementBackground", 11},
    {"textColorHighlightInverse", 11},
    {"textColorLinkInverse", 11},
    {"editTextColor", 11},
    {"editTextBackground", 11},
    {"horizontalScrollViewStyle", 11},
    {"layerType", 11},
    {"alertDialogIcon", 11},
    {"windowMinWidthMajor", 11},
    {"windowMinWidthMinor", 11},
    {"queryHint", 11},
    {"fastScrollTextColor", 11},
    {"largeHeap", 11},
    {"windowCloseOnTouchOutside", 11},
    {"datePickerStyle", 11},
    {"calendarViewStyle", 11},
    {"textEditSidePasteWindowLayout", 11},
    {"textEditSideNoPasteWindowLayout", 11},
    {"actionMenuTextAppearance", 11},
    {"actionMenuTextColor", 11},
    {"textCursorDrawable", 12},
    {"resizeMode", 12},
    {"requiresSmallestWidthDp", 12},
    {"compatibleWidthLimitDp", 12},
    {"largestWidthLimitDp", 12},
    {"state_hovered", 13},
    {"state_drag_can_accept", 13},
    {"state_drag_hovered", 13},
    {"stopWithTask", 13},
    {"switchTextOn", 13},
    {"switchTextOff", 13},
    {"switchPreferenceStyle", 13},
    {"switchTextAppearance", 13},
    {"track", 13},
    {"switchMinWidth", 13},
    {"switchPadding", 13},
    {"thumbTextPadding", 13},
    {"textSuggestionsWindowStyle", 13},
    {"textEditSuggestionItemLayout", 13},
    {"rowCount", 13},
    {"rowOrderPreserved", 13},
    {"columnCount", 13},
    {"columnOrderPreserved", 13},
    {"useDefaultMargins", 13},
    {"alignmentMode", 13},
    {"layout_row", 13},
    {"layout_rowSpan", 13},
    {"layout_columnSpan", 13},
    {"actionModeSelectAllDrawable", 13},
    {"isAuxiliary", 13},
    {"accessibilityEventTypes", 13},
    {"packageNames", 13},
    {"accessibilityFeedbackType", 13},
    {"notificationTimeout", 13},
    {"accessibilityFlags", 13},
    {"canRetrieveWindowContent", 13},
    {"listPreferredItemHeightLarge", 13},
    {"listPreferredItemHeightSmall", 13},
    {"actionBarSplitStyle", 13},
    {"actionProviderClass", 13},
    {"backgroundStacked", 13},
    {"backgroundSplit", 13},
    {"textAllCaps", 13},
    {"colorPressedHighlight", 13},
    {"colorLongPressedHighlight", 13},
    {"colorFocusedHighlight", 13},
    {"colorActivatedHighlight", 13},
    {"colorMultiSelectHighlight", 13},
    {"drawableStart", 13},
    {"drawableEnd", 13},
    {"actionModeStyle", 13},
    {"minResizeWidth", 13},
    {"minResizeHeight", 13},
    {"actionBarWidgetTheme", 13},
    {"uiOptions", 13},
    {"subtypeLocale", 13},
    {"subtypeExtraValue", 13},
    {"actionBarDivider", 13},
    {"actionBarItemBackground", 13},
    {"actionModeSplitBackground", 13},
    {"textAppearanceListItem", 13},
    {"textAppearanceListItemSmall", 13},
    {"targetDescriptions", 13},
    {"directionDescriptions", 13},
    {"overridesImplicitlyEnabledSubtype", 13},
    {"listPreferredItemPaddingLeft", 13},
    {"listPreferredItemPaddingRight", 13},
    {"requiresFadingEdge", 13},
    {"publicKey", 13},
    {"parentActivityName", 16},
    {"isolatedProcess", 16},
    {"importantForAccessibility", 16},
    {"keyboardLayout", 16},
    {"fontFamily", 16},
    {"mediaRouteButtonStyle", 16},
    {"mediaRouteTypes", 16},
    {"supportsRtl", 17},
    {"textDirection", 17},
    {"textAlignment", 17},
    {"layoutDirection", 17},
    {"paddingStart", 17},
    {"paddingEnd", 17},
    {"layout_marginStart", 17},
    {"layout_marginEnd", 17},
    {"layout_toStartOf", 17},
    {"layout_toEndOf", 17},
    {"layout_alignStart", 17},
    {"layout_alignEnd", 17},
    {"layout_alignParentStart", 17},
    {"layout_alignParentEnd", 17},
    {"listPreferredItemPaddingStart", 17},
    {"listPreferredItemPaddingEnd", 17},
    {"singleUser", 17},
    {"presentationTheme", 17},
    {"subtypeId", 17},
    {"initialKeyguardLayout", 17},
    {"widgetCategory", 17},
    {"permissionGroupFlags", 17},
    {"labelFor", 17},
    {"permissionFlags", 17},
    {"checkedTextViewStyle", 17},
    {"showOnLockScreen", 17},
    {"format12Hour", 17},
    {"format24Hour", 17},
    {"timeZone", 17},
    {"mipMap", 18},
    {"mirrorForRtl", 18},
    {"windowOverscan", 18},
    {"requiredForAllUsers", 18},
    {"indicatorStart", 18},
    {"indicatorEnd", 18},
    {"childIndicatorStart", 18},
    {"childIndicatorEnd", 18},
    {"restrictedAccountType", 18},
    {"requiredAccountType", 18},
    {"canRequestTouchExplorationMode", 18},
    {"canRequestEnhancedWebAccessibility", 18},
    {"canRequestFilterKeyEvents", 18},
    {"layoutMode", 18},
    {"keySet", 19},
    {"targetId", 19},
    {"fromScene", 19},
    {"toScene", 19},
    {"transition", 19},
    {"transitionOrdering", 19},
    {"fadingMode", 19},
    {"startDelay", 19},
    {"ssp", 19},
    {"sspPrefix", 19},
    {"sspPattern", 19},
    {"addPrintersActivity", 19},
    {"vendor", 19},
    {"category", 19},
    {"isAsciiCapable", 19},
    {"autoMirrored", 19},
    {"supportsSwitchingToNextInputMethod", 19},
    {"requireDeviceUnlock", 19},
    {"apduServiceBanner", 19},
    {"accessibilityLiveRegion", 19},
    {"windowTranslucentStatus", 19},
    {"windowTranslucentNavigation", 19},
    {"advancedPrintOptionsActivity", 19},
    {"banner", 20},
    {"windowSwipeToDismiss", 20},
    {"isGame", 20},
    {"allowEmbedded", 20},
    {"setupActivity", 20},
    {"fastScrollStyle", 21},
    {"windowContentTransitions", 21},
    {"windowContentTransitionManager", 21},
    {"translationZ", 21},
    {"tintMode", 21},
    {"controlX1", 21},
    {"controlY1", 21},
    {"controlX2", 21},
    {"controlY2", 21},
    {"transitionName", 21},
    {"transitionGroup", 21},
    {"viewportWidth", 21},
    {"viewportHeight", 21},
    {"fillColor", 21},
    {"pathData", 21},
    {"strokeColor", 21},
    {"strokeWidth", 21},
    {"trimPathStart", 21},
    {"trimPathEnd", 21},
    {"trimPathOffset", 21},
    {"strokeLineCap", 21},
    {"strokeLineJoin", 21},
    {"strokeMiterLimit", 21},
    {"colorControlNormal", 21},
    {"colorControlActivated", 21},
    {"colorButtonNormal", 21},
    {"colorControlHighlight", 21},
    {"persistableMode", 21},
    {"titleTextAppearance", 21},
    {"subtitleTextAppearance", 21},
    {"slideEdge", 21},
    {"actionBarTheme", 21},
    {"textAppearanceListItemSecondary", 21},
    {"colorPrimary", 21},
    {"colorPrimaryDark", 21},
    {"colorAccent", 21},
    {"nestedScrollingEnabled", 21},
    {"windowEnterTransition", 21},
    {"windowExitTransition", 21},
    {"windowSharedElementEnterTransition", 21},
    {"windowSharedElementExitTransition", 21},
    {"windowAllowReturnTransitionOverlap", 21},
    {"windowAllowEnterTransitionOverlap", 21},
    {"sessionService", 21},
    {"stackViewStyle", 21},
    {"switchStyle", 21},
    {"elevation", 21},
    {"excludeId", 21},
    {"excludeClass", 21},
    {"hideOnContentScroll", 21},
    {"actionOverflowMenuStyle", 21},
    {"documentLaunchMode", 21},
    {"maxRecents", 21},
    {"autoRemoveFromRecents", 21},
    {"stateListAnimator", 21},
    {"toId", 21},
    {"fromId", 21},
    {"reversible", 21},
    {"splitTrack", 21},
    {"targetName", 21},
    {"excludeName", 21},
    {"matchOrder", 21},
    {"windowDrawsSystemBarBackgrounds", 21},
    {"statusBarColor", 21},
    {"navigationBarColor", 21},
    {"contentInsetStart", 21},
    {"contentInsetEnd", 21},
    {"contentInsetLeft", 21},
    {"contentInsetRight", 21},
    {"paddingMode", 21},
    {"layout_rowWeight", 21},
    {"layout_columnWeight", 21},
    {"translateX", 21},
    {"translateY", 21},
    {"selectableItemBackgroundBorderless", 21},
    {"elegantTextHeight", 21},
    {"searchKeyphraseId", 21},
    {"searchKeyphrase", 21},
    {"searchKeyphraseSupportedLocales", 21},
    {"windowTransitionBackgroundFadeDuration", 21},
    {"overlapAnchor", 21},
    {"progressTint", 21},
    {"progressTintMode", 21},
    {"progressBackgroundTint", 21},
    {"progressBackgroundTintMode", 21},
    {"secondaryProgressTint", 21},
    {"secondaryProgressTintMode", 21},
    {"indeterminateTint", 21},
    {"indeterminateTintMode", 21},
    {"backgroundTint", 21},
    {"backgroundTintMode", 21},
    {"foregroundTint", 21},
    {"foregroundTintMode", 21},
    {"buttonTint", 21},
    {"buttonTintMode", 21},
    {"thumbTint", 21},
    {"thumbTintMode", 21},
    {"fullBackupOnly", 21},
    {"propertyXName", 21},
    {"propertyYName", 21},
    {"relinquishTaskIdentity", 21},
    {"tileModeX", 21},
    {"tileModeY", 21},
    {"actionModeShareDrawable", 21},
    {"actionModeFindDrawable", 21},
    {"actionModeWebSearchDrawable", 21},
    {"transitionVisibilityMode", 21},
    {"minimumHorizontalAngle", 21},
    {"minimumVerticalAngle", 21},
    {"maximumAngle", 21},
    {"searchViewStyle", 21},
    {"closeIcon", 21},
    {"goIcon", 21},
    {"searchIcon", 21},
    {"voiceIcon", 21},
    {"commitIcon", 21},
    {"suggestionRowLayout", 21},
    {"queryBackground", 21},
    {"submitBackground", 21},
    {"buttonBarPositiveButtonStyle", 21},
    {"buttonBarNeutralButtonStyle", 21},
    {"buttonBarNegativeButtonStyle", 21},
    {"popupElevation", 21},
    {"actionBarPopupTheme", 21},
    {"multiArch", 21},
    {"touchscreenBlocksFocus", 21},
    {"windowElevation", 21},
    {"launchTaskBehindTargetAnimation", 21},
    {"launchTaskBehindSourceAnimation", 21},
    {"restrictionType", 21},
    {"dayOfWeekBackground", 21},
    {"dayOfWeekTextAppearance", 21},
    {"headerMonthTextAppearance", 21},
    {"headerDayOfMonthTextAppearance", 21},
    {"headerYearTextAppearance", 21},
    {"yearListItemTextAppearance", 21},
    {"yearListSelectorColor", 21},
    {"calendarTextColor", 21},
    {"recognitionService", 21},
    {"timePickerStyle", 21},
    {"timePickerDialogTheme", 21},
    {"headerTimeTextAppearance", 21},
    {"headerAmPmTextAppearance", 21},
    {"numbersTextColor", 21},
    {"numbersBackgroundColor", 21},
    {"numbersSelectorColor", 21},
    {"amPmTextColor", 21},
    {"amPmBackgroundColor", 21},
    {"searchKeyphraseRecognitionFlags", 21},
    {"checkMarkTint", 21},
    {"checkMarkTintMode", 21},
    {"popupTheme", 21},
    {"toolbarStyle", 21},
    {"windowClipToOutline", 21},
    {"datePickerDialogTheme", 21},
    {"showText", 21},
    {"windowReturnTransition", 21},
    {"windowReenterTransition", 21},
    {"windowSharedElementReturnTransition", 21},
    {"windowSharedElementReenterTransition", 21},
    {"resumeWhilePausing", 21},
    {"datePickerMode", 21},
    {"timePickerMode", 21},
    {"inset", 21},
    {"letterSpacing", 21},
    {"fontFeatureSettings", 21},
    {"outlineProvider", 21},
    {"contentAgeHint", 21},
    {"country", 21},
    {"windowSharedElementsUseOverlay", 21},
    {"reparent", 21},
    {"reparentWithOverlay", 21},
    {"ambientShadowAlpha", 21},
    {"spotShadowAlpha", 21},
    {"navigationIcon", 21},
    {"navigationContentDescription", 21},
    {"fragmentExitTransition", 21},
    {"fragmentEnterTransition", 21},
    {"fragmentSharedElementEnterTransition", 21},
    {"fragmentReturnTransition", 21},
    {"fragmentSharedElementReturnTransition", 21},
    {"fragmentReenterTransition", 21},
    {"fragmentAllowEnterTransitionOverlap", 21},
    {"fragmentAllowReturnTransitionOverlap", 21},
    {"patternPathData", 21},
    {"strokeAlpha", 21},
    {"fillAlpha", 21},
    {"windowActivityTransitions", 21},
    {"colorEdgeEffect", 21}};

size_t FindAttributeSdkLevel(const ResourceName& name) {
  if (name.package != "android" && name.type != ResourceType::kAttr) {
    return 0;
  }

  auto iter = sAttrMap.find(name.entry);
  if (iter != sAttrMap.end()) {
    return iter->second;
  }
  return SDK_LOLLIPOP_MR1;
}

std::pair<StringPiece, int> GetDevelopmentSdkCodeNameAndVersion() {
  return std::make_pair(StringPiece(sDevelopmentSdkCodeName),
                        sDevelopmentSdkLevel);
}

}  // namespace aapt
