/*
 * Copyright (C) 2010 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.internal.view.menu;

import android.content.Context;
import android.graphics.drawable.Drawable;
import android.util.AttributeSet;
import android.view.SoundEffectConstants;
import android.view.View;
import android.widget.ImageButton;

/**
 * @hide
 */
public class ActionMenuItemView extends ImageButton implements MenuView.ItemView {
    private static final String TAG = "ActionMenuItemView";
    
    private MenuItemImpl mItemData;
    private CharSequence mTitle;
    private MenuBuilder.ItemInvoker mItemInvoker;
    
    public ActionMenuItemView(Context context) {
        this(context, null);
    }
    
    public ActionMenuItemView(Context context, AttributeSet attrs) {
        this(context, attrs, com.android.internal.R.attr.actionButtonStyle);
    }
    
    public ActionMenuItemView(Context context, AttributeSet attrs, int defStyle) {
        super(context, attrs, defStyle);
    }

    public MenuItemImpl getItemData() {
        return mItemData;
    }

    public void initialize(MenuItemImpl itemData, int menuType) {
        mItemData = itemData;
        
        setClickable(true);
        setFocusable(true);
        setTitle(itemData.getTitle());
        setIcon(itemData.getIcon());
        setId(itemData.getItemId());
        
        setVisibility(itemData.isVisible() ? View.VISIBLE : View.GONE);
        setEnabled(itemData.isEnabled());
    }
    
    @Override
    public boolean performClick() {
        // Let the view's listener have top priority
        if (super.performClick()) {
            return true;
        }
        
        if (mItemInvoker != null && mItemInvoker.invokeItem(mItemData)) {
            playSoundEffect(SoundEffectConstants.CLICK);
            return true;
        } else {
            return false;
        }
    }
    
    public void setItemInvoker(MenuBuilder.ItemInvoker invoker) {
        mItemInvoker = invoker;
    }

    public boolean prefersCondensedTitle() {
        return false;
    }

    public void setCheckable(boolean checkable) {
        // TODO Support checkable action items
    }

    public void setChecked(boolean checked) {
        // TODO Support checkable action items
    }

    public void setIcon(Drawable icon) {
        setImageDrawable(icon);
    }

    public void setShortcut(boolean showShortcut, char shortcutKey) {
        // Action buttons don't show text for shortcut keys.
    }

    public void setTitle(CharSequence title) {
        mTitle = title;
    }

    public boolean showsIcon() {
        return true;
    }

}
