/*
 * Copyright (C) 2007 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.util;

/**
 * Reimplements _Original_FloatMath with the standard libraries.
 * 
 * Math routines similar to those found in {@link java.lang.Math}. Performs
 * computations on {@code float} values directly without incurring the overhead
 * of conversions to and from {@code double}.
 *
 * <p>On one platform, {@code FloatMath.sqrt(100)} executes in one third of the
 * time required by {@code java.lang.Math.sqrt(100)}.</p>
 */
public class FloatMath {

    /** Prevents instantiation. */
    private FloatMath() {}

    /**
     * Returns the float conversion of the most positive (i.e. closest to
     * positive infinity) integer value which is less than the argument.
     *
     * @param value to be converted
     * @return the floor of value
     */
    public static float floor(float value) {
        return (float)Math.floor(value);
    }

    /**
     * Returns the float conversion of the most negative (i.e. closest to
     * negative infinity) integer value which is greater than the argument.
     *
     * @param value to be converted
     * @return the ceiling of value
     */
    public static float ceil(float value) {
        return (float)Math.ceil(value);
    }

    /**
     * Returns the closest float approximation of the sine of the argument.
     *
     * @param angle to compute the cosine of, in radians
     * @return the sine of angle
     */
    public static  float sin(float angle) {
        return (float)Math.sin(angle);
    }

    /**
     * Returns the closest float approximation of the cosine of the argument.
     *
     * @param angle to compute the cosine of, in radians
     * @return the cosine of angle
     */
    public static float cos(float angle) {
        return (float)Math.cos(angle);
    }

    /**
     * Returns the closest float approximation of the square root of the
     * argument.
     *
     * @param value to compute sqrt of
     * @return the square root of value
     */
    public static float sqrt(float value) {
        return (float)Math.sqrt(value);
    }
}
