/*
 * Copyright (C) 2008 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.internal.os;

import junit.framework.TestCase;

import java.util.Arrays;
import java.util.List;
import java.util.ArrayList;
import java.util.Collections;
import java.io.StringWriter;
import java.io.PrintWriter;

public class LoggingPrintStreamTest extends TestCase {

    TestPrintStream out = new TestPrintStream();

    public void testPrintException() {
        @SuppressWarnings("ThrowableInstanceNeverThrown")
        Throwable t = new Throwable("Ignore me.");

        StringWriter sout = new StringWriter();
        t.printStackTrace(new PrintWriter(sout));

        t.printStackTrace(out);
        // t.printStackTrace();

        String[] lines = sout.toString().split("\\n");
        assertEquals(Arrays.asList(lines), out.lines);
    }

    public void testPrintObject() {
        Object o = new Object();
        out.print(4);
        out.print(o);
        out.print(2);
        out.flush();
        assertEquals(Arrays.asList("4" + o + "2"), out.lines);
    }

    public void testPrintlnObject() {
        Object o = new Object();
        out.print(4);
        out.println(o);
        out.print(2);
        out.flush();
        assertEquals(Arrays.asList("4" + o, "2"), out.lines);
    }

    public void testPrintf() {
        out.printf("Name: %s\nEmployer: %s", "Bob", "Google");
        assertEquals(Arrays.asList("Name: Bob"), out.lines);
        out.flush();
        assertEquals(Arrays.asList("Name: Bob", "Employer: Google"), out.lines);
    }

    public void testPrintInt() {
        out.print(4);
        out.print(2);
        assertTrue(out.lines.isEmpty());
        out.flush();
        assertEquals(Collections.singletonList("42"), out.lines);
    }

    public void testPrintlnInt() {
        out.println(4);
        out.println(2);
        assertEquals(Arrays.asList("4", "2"), out.lines);
    }

    public void testPrintCharArray() {
        out.print("Foo\nBar\nTee".toCharArray());
        assertEquals(Arrays.asList("Foo", "Bar"), out.lines);
        out.flush();
        assertEquals(Arrays.asList("Foo", "Bar", "Tee"), out.lines);
    }

    public void testPrintString() {
        out.print("Foo\nBar\nTee");
        assertEquals(Arrays.asList("Foo", "Bar"), out.lines);
        out.flush();
        assertEquals(Arrays.asList("Foo", "Bar", "Tee"), out.lines);
    }

    public void testPrintlnCharArray() {
        out.println("Foo\nBar\nTee".toCharArray());
        assertEquals(Arrays.asList("Foo", "Bar", "Tee"), out.lines);
    }

    public void testPrintlnString() {
        out.println("Foo\nBar\nTee");
        assertEquals(Arrays.asList("Foo", "Bar", "Tee"), out.lines);
    }

    public void testPrintlnStringWithBufferedData() {
        out.print(5);
        out.println("Foo\nBar\nTee");
        assertEquals(Arrays.asList("5Foo", "Bar", "Tee"), out.lines);
    }

    public void testAppend() {
        out.append("Foo\n")
            .append('4')
            .append('\n')
            .append("Bar", 1, 2)
            .append('\n');
        assertEquals(Arrays.asList("Foo", "4", "a"), out.lines);
    }

    static class TestPrintStream extends LoggingPrintStream {

        final List<String> lines = new ArrayList<String>();

        protected void log(String line) {
            lines.add(line);
        }
    }

}
