/*
 * Copyright (C) 2010 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.systemui.statusbar.tablet;

import java.util.ArrayList;
import java.util.List;

import android.animation.Animator;
import android.animation.AnimatorSet;
import android.animation.ObjectAnimator;
import android.app.ActivityManager;
import android.content.Context;
import android.content.Intent;
import android.content.pm.ActivityInfo;
import android.content.pm.PackageManager;
import android.content.pm.ResolveInfo;
import android.content.res.Configuration;
import android.content.res.Resources;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.graphics.Matrix;
import android.graphics.Paint;
import android.graphics.Rect;
import android.graphics.RectF;
import android.graphics.Shader.TileMode;
import android.graphics.drawable.BitmapDrawable;
import android.graphics.drawable.Drawable;
import android.util.AttributeSet;
import android.util.DisplayMetrics;
import android.util.Log;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.animation.DecelerateInterpolator;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;

import com.android.systemui.R;

public class RecentAppsPanel extends LinearLayout implements StatusBarPanel, OnClickListener {
    private static final int GLOW_PADDING = 15;
    private static final String TAG = "RecentAppsPanel";
    private static final boolean DEBUG = TabletStatusBar.DEBUG;
    private static final int DISPLAY_TASKS_PORTRAIT = 7; // Limited by max binder transaction size
    private static final int DISPLAY_TASKS_LANDSCAPE = 5; // number of recent tasks to display
    private static final int MAX_TASKS = DISPLAY_TASKS_PORTRAIT + 1; // allow extra for non-apps
    private static final int STAGGER_ANIMATION_DELAY = 30;
    private static final long ALPHA_ANIMATION_DURATION = 120;
    private TabletStatusBar mBar;
    private LinearLayout mRecentsContainer;
    private View mRecentsGlowView;
    private ArrayList<ActivityDescription> mActivityDescriptions;
    private int mIconDpi;
    private AnimatorSet mAnimationSet;
    private View mBackgroundProtector;
    private Bitmap mGlowBitmap;

    static class ActivityDescription {
        int id;
        Bitmap thumbnail; // generated by Activity.onCreateThumbnail()
        Drawable icon; // application package icon
        String label; // application package label
        CharSequence description; // generated by Activity.onCreateDescription()
        Intent intent; // launch intent for application
        Matrix matrix; // arbitrary rotation matrix to correct orientation
        String packageName; // used to override animations (see onClick())
        int position; // position in list

        public ActivityDescription(Bitmap _thumbnail,
                Drawable _icon, String _label, CharSequence _desc, Intent _intent,
                int _id, int _pos, String _packageName)
        {
            thumbnail = _thumbnail;
            icon = _icon;
            label = _label;
            description = _desc;
            intent = _intent;
            id = _id;
            position = _pos;
            packageName = _packageName;
        }
    };

    public boolean isInContentArea(int x, int y) {
        final int l = mRecentsContainer.getPaddingLeft();
        final int r = mRecentsContainer.getWidth() - mRecentsContainer.getPaddingRight();
        final int t = mRecentsContainer.getPaddingTop();
        final int b = mRecentsContainer.getHeight() - mRecentsContainer.getPaddingBottom();
        return x >= l && x < r && y >= t && y < b;
    }

    public void setBar(TabletStatusBar bar) {
        mBar = bar;
    }

    public RecentAppsPanel(Context context, AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public RecentAppsPanel(Context context, AttributeSet attrs, int defStyle) {
        super(context, attrs, defStyle);

        Resources res = context.getResources();
        boolean xlarge = (res.getConfiguration().screenLayout
                & Configuration.SCREENLAYOUT_SIZE_MASK) == Configuration.SCREENLAYOUT_SIZE_XLARGE;

        mIconDpi = xlarge ? DisplayMetrics.DENSITY_HIGH : res.getDisplayMetrics().densityDpi;
        mGlowBitmap = BitmapFactory.decodeResource(res, R.drawable.recents_thumbnail_bg);
    }

    @Override
    protected void onFinishInflate() {
        super.onFinishInflate();
        mRecentsContainer = (LinearLayout) findViewById(R.id.recents_container);
        mRecentsGlowView = findViewById(R.id.recents_glow);
        mBackgroundProtector = (View) findViewById(R.id.recents_bg_protect);

        // In order to save space, we make the background texture repeat in the Y direction
        View view = findViewById(R.id.recents_bg_protect);
        if (view != null && view.getBackground() instanceof BitmapDrawable) {
            ((BitmapDrawable) view.getBackground()).setTileModeY(TileMode.REPEAT);
        }
    }

    @Override
    protected void onConfigurationChanged(Configuration newConfig) {
        super.onConfigurationChanged(newConfig);
        // we show more in portrait mode, so update UI if orientation changes
        updateUiElements(newConfig, false);
    }

    @Override
    protected void onVisibilityChanged(View changedView, int visibility) {
        super.onVisibilityChanged(changedView, visibility);
        if (DEBUG) Log.v(TAG, "onVisibilityChanged(" + changedView + ", " + visibility + ")");
        if (visibility == View.VISIBLE && changedView == this) {
            refreshApplicationList();
            mRecentsContainer.setScrollbarFadingEnabled(true);
            mRecentsContainer.scrollTo(0, 0);
        }
    }

    private Drawable getFullResDefaultActivityIcon() {
        return getFullResIcon(Resources.getSystem(),
                com.android.internal.R.drawable.sym_def_app_icon);
    }

    private Drawable getFullResIcon(Resources resources, int iconId) {
        return resources.getDrawableForDensity(iconId, mIconDpi);
    }

    private Drawable getFullResIcon(ResolveInfo info, PackageManager packageManager) {
        Resources resources;
        try {
            resources = packageManager.getResourcesForApplication(
                    info.activityInfo.applicationInfo);
        } catch (PackageManager.NameNotFoundException e) {
            resources = null;
        }
        if (resources != null) {
            int iconId = info.activityInfo.getIconResource();
            if (iconId != 0) {
                return getFullResIcon(resources, iconId);
            }
        }
        return getFullResDefaultActivityIcon();
    }

    private ArrayList<ActivityDescription> getRecentTasks() {
        ArrayList<ActivityDescription> activityDescriptions = new ArrayList<ActivityDescription>();
        final PackageManager pm = mContext.getPackageManager();
        final ActivityManager am = (ActivityManager)
                mContext.getSystemService(Context.ACTIVITY_SERVICE);

        final List<ActivityManager.RecentTaskInfo> recentTasks =
                am.getRecentTasks(MAX_TASKS, ActivityManager.RECENT_IGNORE_UNAVAILABLE
                        | ActivityManager.TASKS_GET_THUMBNAILS);

        ActivityInfo homeInfo = new Intent(Intent.ACTION_MAIN).addCategory(Intent.CATEGORY_HOME)
                    .resolveActivityInfo(pm, 0);

        int numTasks = recentTasks.size();

        // skip the first activity - assume it's either the home screen or the current app.
        final int first = 1;
        for (int i = first, index = 0; i < numTasks && (index < MAX_TASKS); ++i) {
            final ActivityManager.RecentTaskInfo recentInfo = recentTasks.get(i);

            Intent intent = new Intent(recentInfo.baseIntent);
            if (recentInfo.origActivity != null) {
                intent.setComponent(recentInfo.origActivity);
            }

            // Skip the current home activity.
            if (homeInfo != null
                    && homeInfo.packageName.equals(intent.getComponent().getPackageName())
                    && homeInfo.name.equals(intent.getComponent().getClassName())) {
                continue;
            }

            intent.setFlags((intent.getFlags()&~Intent.FLAG_ACTIVITY_RESET_TASK_IF_NEEDED)
                    | Intent.FLAG_ACTIVITY_NEW_TASK);
            final ResolveInfo resolveInfo = pm.resolveActivity(intent, 0);
            if (resolveInfo != null) {
                final ActivityInfo info = resolveInfo.activityInfo;
                final String title = info.loadLabel(pm).toString();
                // Drawable icon = info.loadIcon(pm);
                Drawable icon = getFullResIcon(resolveInfo, pm);
                int id = recentTasks.get(i).id;
                if (title != null && title.length() > 0 && icon != null) {
                    if (DEBUG) Log.v(TAG, "creating activity desc for id=" + id + ", label=" + title);
                    ActivityDescription item = new ActivityDescription(
                            recentInfo.thumbnail, icon, title,
                            recentInfo.description, intent, id, index, info.packageName);
                    activityDescriptions.add(item);
                    ++index;
                } else {
                    if (DEBUG) Log.v(TAG, "SKIPPING item " + id);
                }
            }
        }
        return activityDescriptions;
    }

    ActivityDescription findActivityDescription(int id)
    {
        ActivityDescription desc = null;
        for (int i = 0; i < mActivityDescriptions.size(); i++) {
            ActivityDescription item = mActivityDescriptions.get(i);
            if (item != null && item.id == id) {
                desc = item;
                break;
            }
        }
        return desc;
    }

    private void refreshApplicationList() {
        mActivityDescriptions = getRecentTasks();
        updateUiElements(getResources().getConfiguration(), true);
    }

    private Bitmap compositeBitmap(Bitmap background, Bitmap thumbnail) {
        Bitmap outBitmap = background.copy(background.getConfig(), true);
        if (thumbnail != null) {
            Canvas canvas = new Canvas(outBitmap);
            Paint paint = new Paint();
            paint.setAntiAlias(true);
            paint.setFilterBitmap(true);
            paint.setAlpha(255);
            final int srcWidth = thumbnail.getWidth();
            final int height = thumbnail.getHeight();
            final int srcHeight = srcWidth > height ? height
                    : (height - height * srcWidth / height);
            canvas.drawBitmap(thumbnail,
                    new Rect(0, 0, srcWidth-1, srcHeight-1),
                    new RectF(GLOW_PADDING,
                            GLOW_PADDING - 4.0f,
                            outBitmap.getWidth() - GLOW_PADDING + 2.0f,
                            outBitmap.getHeight() - GLOW_PADDING + 3.0f), paint);
        }
        return outBitmap;
    }

    private void updateUiElements(Configuration config, boolean animate) {
        mRecentsContainer.removeAllViews();

        final float initialAlpha = 0.0f;
        final int first = 0;
        final boolean isPortrait = config.orientation == Configuration.ORIENTATION_PORTRAIT;
        final int taskCount = isPortrait ? DISPLAY_TASKS_PORTRAIT : DISPLAY_TASKS_LANDSCAPE;
        final int last = Math.min(mActivityDescriptions.size(), taskCount) - 1;
        ArrayList<Animator> anims = new ArrayList<Animator>(last+1);
        DecelerateInterpolator interp = new DecelerateInterpolator();
        for (int i = last; i >= first; i--) {
            ActivityDescription activityDescription = mActivityDescriptions.get(i);
            View view = View.inflate(mContext, R.layout.status_bar_recent_item, null);
            ImageView appThumbnail = (ImageView) view.findViewById(R.id.app_thumbnail);
            ImageView appIcon = (ImageView) view.findViewById(R.id.app_icon);
            TextView appLabel = (TextView) view.findViewById(R.id.app_label);
            TextView appDesc = (TextView) view.findViewById(R.id.app_description);
            final Bitmap thumb = activityDescription.thumbnail;
            appThumbnail.setImageBitmap(compositeBitmap(mGlowBitmap, thumb));
            appIcon.setImageDrawable(activityDescription.icon);
            appLabel.setText(activityDescription.label);
            appDesc.setText(activityDescription.description);
            view.setOnClickListener(this);
            view.setTag(activityDescription);
            mRecentsContainer.addView(view);

            if (animate) {
                view.setAlpha(initialAlpha);
                ObjectAnimator anim = ObjectAnimator.ofFloat(view, "alpha", initialAlpha, 1.0f);
                anim.setDuration(ALPHA_ANIMATION_DURATION);
                anim.setStartDelay((last-i) * STAGGER_ANIMATION_DELAY);
                anim.setInterpolator(interp);
                anims.add(anim);
            }
        }

        int views = mRecentsContainer.getChildCount();
        mRecentsContainer.setVisibility(views > 0 ? View.VISIBLE : View.GONE);
        mRecentsGlowView.setVisibility(views > 0 ? View.VISIBLE : View.GONE);

        if (animate && views > 0) {
            ObjectAnimator anim = ObjectAnimator.ofFloat(mRecentsGlowView, "alpha",
                    initialAlpha, 1.0f);
            anim.setDuration((last-first) * STAGGER_ANIMATION_DELAY);
            anim.setInterpolator(interp);
            anims.add(anim);

            anim = ObjectAnimator.ofFloat(mBackgroundProtector, "alpha",
                    initialAlpha, 1.0f);
            anim.setDuration(last * STAGGER_ANIMATION_DELAY);
            anim.setInterpolator(interp);
            anims.add(anim);
        }

        if (anims.size() > 0) {
            mAnimationSet = new AnimatorSet();
            mAnimationSet.playTogether(anims);
            mAnimationSet.start();
        }
    }

    public void onClick(View v) {
        ActivityDescription ad = (ActivityDescription)v.getTag();
        final ActivityManager am = (ActivityManager)
                getContext().getSystemService(Context.ACTIVITY_SERVICE);
        if (ad.id >= 0) {
            // This is an active task; it should just go to the foreground.
            am.moveTaskToFront(ad.id, ActivityManager.MOVE_TASK_WITH_HOME);
        } else {
            Intent intent = ad.intent;
            intent.addFlags(Intent.FLAG_ACTIVITY_LAUNCHED_FROM_HISTORY
                    | Intent.FLAG_ACTIVITY_TASK_ON_HOME);
            if (DEBUG) Log.v(TAG, "Starting activity " + intent);
            getContext().startActivity(intent);
        }
        mBar.animateCollapse();
    }
}
