/*
 * Copyright (C) 2016 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License
 */

package com.android.incallui.ringtone;

import com.google.common.base.Preconditions;

import android.net.Uri;
import android.support.annotation.Nullable;

import com.android.contacts.common.compat.CompatUtils;
import com.android.contacts.common.testing.NeededForTesting;
import com.android.incallui.Call;
import com.android.incallui.Call.State;
import com.android.incallui.CallList;

/**
 * Class that determines when ringtones should be played and can play the call waiting tone when
 * necessary.
 */
public class DialerRingtoneManager {

    /*
     * Flag used to determine if the Dialer is responsible for playing ringtones for incoming calls.
     * Once we're ready to enable Dialer Ringing, these flags should be removed.
     */
    private static final boolean IS_DIALER_RINGING_ENABLED = true;
    private Boolean mIsDialerRingingEnabledForTesting;

    private final InCallTonePlayer mInCallTonePlayer;
    private final CallList mCallList;

    /**
     * Creates the DialerRingtoneManager with the given {@link InCallTonePlayer}.
     *
     * @param inCallTonePlayer the tone player used to play in-call tones.
     * @param callList the CallList used to check for {@link State#CALL_WAITING}
     * @throws NullPointerException if inCallTonePlayer or callList are null
     */
    public DialerRingtoneManager(InCallTonePlayer inCallTonePlayer, CallList callList) {
        mInCallTonePlayer = Preconditions.checkNotNull(inCallTonePlayer);
        mCallList = Preconditions.checkNotNull(callList);
    }

    /**
     * Determines if a ringtone should be played for the given call state (see {@link State}) and
     * {@link Uri}.
     *
     * @param callState the call state for the call being checked.
     * @param ringtoneUri the ringtone to potentially play.
     * @return {@code true} if the ringtone should be played, {@code false} otherwise.
     */
    public boolean shouldPlayRingtone(int callState, @Nullable Uri ringtoneUri) {
        return isDialerRingingEnabled()
                && translateCallStateForCallWaiting(callState) == State.INCOMING
                && ringtoneUri != null;
    }

    /**
     * The incoming callState is never set as {@link State#CALL_WAITING} because
     * {@link Call#translateState(int)} doesn't account for that case, check for it here
     */
    private int translateCallStateForCallWaiting(int callState) {
        if (callState != State.INCOMING) {
            return callState;
        }
        return mCallList.getActiveCall() == null ? State.INCOMING : State.CALL_WAITING;
    }

    private boolean isDialerRingingEnabled() {
        if (mIsDialerRingingEnabledForTesting != null) {
            return mIsDialerRingingEnabledForTesting;
        }
        return CompatUtils.isNCompatible() && IS_DIALER_RINGING_ENABLED;
    }

    /**
     * Determines if a call waiting tone should be played for the the given call state
     * (see {@link State}).
     *
     * @param callState the call state for the call being checked.
     * @return {@code true} if the call waiting tone should be played, {@code false} otherwise.
     */
    public boolean shouldPlayCallWaitingTone(int callState) {
        return isDialerRingingEnabled()
                && translateCallStateForCallWaiting(callState) == State.CALL_WAITING
                && !mInCallTonePlayer.isPlayingTone();
    }

    /**
     * Plays the call waiting tone.
     */
    public void playCallWaitingTone() {
        if (!isDialerRingingEnabled()) {
            return;
        }
        mInCallTonePlayer.play(InCallTonePlayer.TONE_CALL_WAITING);
    }

    /**
     * Stops playing the call waiting tone.
     */
    public void stopCallWaitingTone() {
        if (!isDialerRingingEnabled()) {
            return;
        }
        mInCallTonePlayer.stop();
    }

    @NeededForTesting
    void setDialerRingingEnabledForTesting(boolean status) {
        mIsDialerRingingEnabledForTesting = status;
    }
}
