/*
 * Copyright (C) 2009 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.android.providers.contacts;

import android.content.Context;
import android.database.Cursor;
import android.database.DatabaseUtils;
import android.database.sqlite.SQLiteDatabase;
import android.net.Uri;
import android.os.FileUtils;
import android.provider.CallLog;
import android.provider.CallLog.Calls;
import android.provider.ContactsContract.Contacts;
import android.provider.ContactsContract.Data;
import android.provider.ContactsContract.Groups;
import android.provider.ContactsContract.RawContacts;
import android.test.mock.MockContext;
import android.test.suitebuilder.annotation.LargeTest;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;

/**
 * Unit tests for {@link LegacyContactImporter}.
 *
 * Run the test like this:
 * <code>
 * adb shell am instrument -w \
 *         com.android.providers.contacts.tests/android.test.InstrumentationTestRunner
 * </code>
 */
@LargeTest
public class LegacyContactImporterTest extends BaseContactsProvider2Test {

    private static class LegacyMockContext extends MockContext {

        private String mFileName;

        public LegacyMockContext(String fileName) {
            mFileName = fileName;
        }

        @Override
        public SQLiteDatabase openOrCreateDatabase(String file, int mode,
                SQLiteDatabase.CursorFactory factory) {
            return SQLiteDatabase.openDatabase(mFileName, factory, SQLiteDatabase.OPEN_READONLY);
        }

        @Override
        public File getDatabasePath(String name) {
            return new File(mFileName);
        }
    }

    private LegacyMockContext createLegacyMockContext() throws IOException {
        Context context = getTestContext();
        InputStream input = context.getAssets().open("legacy_contacts.db");
        File tempDb = new File(context.getFilesDir(), "legacy_contacts.db");
        if (!FileUtils.copyToFile(input, tempDb)) {
            fail("Could not create test contacts database");
        }

        return new LegacyMockContext(tempDb.getPath());
    }

    @Override
    protected void setUp() throws Exception {
        SynchronousContactsProvider2.resetOpenHelper();
        super.setUp();
        addProvider(TestCallLogProvider.class, CallLog.AUTHORITY);
    }

    public void testContactImport() throws Exception {
        ContactsProvider2 provider = (ContactsProvider2)getProvider();
        LegacyContactImporter importer =
                new LegacyContactImporter(createLegacyMockContext(), provider);
        provider.importLegacyContacts(importer);

        assertQueryResults("expected_groups.txt", Groups.CONTENT_URI, new String[]{
                Groups._ID,
                Groups.ACCOUNT_NAME,
                Groups.ACCOUNT_TYPE,
                Groups.DIRTY,
                Groups.GROUP_VISIBLE,
                Groups.NOTES,
                Groups.RES_PACKAGE,
                Groups.SOURCE_ID,
                Groups.SYSTEM_ID,
                Groups.TITLE,
                Groups.VERSION,
                Groups.SYNC1,
                Groups.SYNC2,
                Groups.SYNC3,
                Groups.SYNC4,
        });

        assertQueryResults("expected_contacts.txt", Contacts.CONTENT_URI, new String[]{
                Contacts._ID,
                Contacts.DISPLAY_NAME,
                Contacts.PHOTO_ID,
                Contacts.PRIMARY_PHONE_ID,
                Contacts.PRIMARY_EMAIL_ID,
                Contacts.TIMES_CONTACTED,
                Contacts.LAST_TIME_CONTACTED,
                Contacts.CUSTOM_RINGTONE,
                Contacts.SEND_TO_VOICEMAIL,
                Contacts.STARRED,
                Contacts.IN_VISIBLE_GROUP,
        });

        assertQueryResults("expected_raw_contacts.txt", RawContacts.CONTENT_URI, new String[]{
                RawContacts._ID,
                RawContacts.ACCOUNT_NAME,
                RawContacts.ACCOUNT_TYPE,
                RawContacts.DELETED,
                RawContacts.DIRTY,
                RawContacts.SOURCE_ID,
                RawContacts.VERSION,
                RawContacts.SYNC1,
                RawContacts.SYNC2,
                RawContacts.SYNC3,
                RawContacts.SYNC4,
        });

        assertQueryResults("expected_data.txt", Data.CONTENT_URI, new String[]{
                Data._ID,
                Data.RAW_CONTACT_ID,
                Data.MIMETYPE,
                Data.DATA1,
                Data.DATA2,
                Data.DATA3,
                Data.DATA4,
                Data.DATA9,
                Data.IS_PRIMARY,
                Data.IS_SUPER_PRIMARY,
                Data.DATA_VERSION,
                Data.SYNC1,
                Data.SYNC2,
                Data.SYNC3,
                Data.SYNC4,
        });

        assertQueryResults("expected_calls.txt", Calls.CONTENT_URI, new String[]{
                Calls._ID,
                Calls.NUMBER,
                Calls.DATE,
                Calls.DURATION,
                Calls.NEW,
                Calls.TYPE,
                Calls.CACHED_NAME,
                Calls.CACHED_NUMBER_LABEL,
                Calls.CACHED_NUMBER_TYPE,
        });
    }

    private void assertQueryResults(String fileName, Uri uri, String[] projection) throws Exception {
        String expected = readAssetAsString(fileName).trim();
        String actual = dumpCursorToString(uri, projection).trim();
        assertEquals("Checking golden file " + fileName, expected, actual);
    }

    private String readAssetAsString(String fileName) throws IOException {
        Context context = getTestContext();
        InputStream input = context.getAssets().open(fileName);
        ByteArrayOutputStream contents = new ByteArrayOutputStream();
        int len;
        byte[] data = new byte[1024];
        do {
            len = input.read(data);
            if (len > 0) contents.write(data, 0, len);
        } while (len == data.length);
        return contents.toString();
    }

    private String dumpCursorToString(Uri uri, String[] projection) {
        Cursor c = mResolver.query(uri, projection, null, null, null);
        if (c == null) {
            return "Null cursor";
        }

        String cursorDump = DatabaseUtils.dumpCursorToString(c);
        c.close();
        return insertLineNumbers(cursorDump);
    }

    private String insertLineNumbers(String multiline) {
        String[] lines = multiline.split("\n");
        StringBuilder sb = new StringBuilder();

        // Ignore the first line that is a volatile header and the last line which is "<<<<<"
        for (int i = 1; i < lines.length - 1; i++) {
            sb.append(i).append(" ").append(lines[i]).append('\n');
        }
        return sb.toString();
    }


    public static class TestCallLogProvider extends CallLogProvider {
        private static OpenHelper mOpenHelper;

        @Override
        protected OpenHelper getOpenHelper(final Context context) {
            if (mOpenHelper == null) {
                mOpenHelper = new OpenHelper(context);
            }
            return mOpenHelper;
        }
    }
}
